// include/indexable_iter.hh
// This file is part of libpbe; see http://svn.chezphil.org/libpbe/
// (C) 2011 Philip Endecott

// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#ifndef libpbe_indexable_iter_hh
#define libpbe_indexable_iter_hh

#include <boost/iterator/iterator_facade.hpp>


// indexable_iter<A> and const_indexable_iter<A> provide iterators over 
// array-like container types A that implement operator[].  The iterators 
// store only a pointer to the container and an index.
//
// Typically a type can provide iterators simply by:
//   typedef indexable_iter<this-type> iterator;
//   typedef const_indexable_iter<this-type> const_iterator;
//
//   iterator begin() { return iterator(this,0); }
//   iterator end()   { return iterator(this,size()); }
//
//   const_iterator begin() const { return const_iterator(this,0); }
//   const_iterator end()   const { return const_iterator(this,size()); }
//
// The implementation uses boost::iterator_facade.


namespace pbe {


template <typename A>
class const_indexable_iter:
  public boost::iterator_facade<const_indexable_iter<A>,
                                const typename A::value_type,
                                boost::random_access_traversal_tag>  // FIXME must ref = const value_type & ?
{
  const A* vec_p;
  size_t index;

  const typename A::value_type& dereference() const { return (*vec_p)[index]; }
  bool equal(const const_indexable_iter& other) const { return vec_p==other.vec_p && index==other.index; }
  void advance(int dist) { index += dist; }
  void increment() { advance(1); }
  void decrement() { advance(-1); }
  int distance_to(const const_indexable_iter& other) const { return other.index - index; }

  friend class boost::iterator_core_access;

public:
  const_indexable_iter(): vec_p(NULL), index(0) {}

  const_indexable_iter(const A* vec_p_, size_t index_):
    vec_p(vec_p_), index(index_) {}

};


template <typename A>
class indexable_iter:
  public boost::iterator_facade<indexable_iter<A>,
                                typename A::value_type,
                                boost::random_access_traversal_tag>
{
  A* vec_p;
  size_t index;

  typename A::value_type& dereference() const { return (*vec_p)[index]; }
  bool equal(const indexable_iter& other) const { return vec_p==other.vec_p && index==other.index; }
  void advance(int dist) { index += dist; }
  void increment() { advance(1); }
  void decrement() { advance(-1); }
  int distance_to(const indexable_iter& other) const { return other.index - index; }

  friend class boost::iterator_core_access;

public:
  indexable_iter(): vec_p(NULL), index(0) {}

  indexable_iter(A* vec_p_, size_t index_):
    vec_p(vec_p_), index(index_) {}

  operator const_indexable_iter<A> () const { return const_indexable_iter<A>(vec_p,index); }
};



};


#endif



