// const_character_iterator.hh
// This file is part of libpbe; see http://anyterm.org/
// (C) 2007-2008 Philip Endecott

// Distributed under the Boost Software License, Version 1.0:
//
// Permission is hereby granted, free of charge, to any person or organization
// obtaining a copy of the software and accompanying documentation covered by
// this license (the "Software") to use, reproduce, display, distribute,
// execute, and transmit the Software, and to prepare derivative works of the
// Software, and to permit third-parties to whom the Software is furnished to
// do so, all subject to the following:
// 
// The copyright notices in the Software and this entire statement, including
// the above license grant, this restriction and the following disclaimer,
// must be included in all copies of the Software, in whole or in part, and
// all derivative works of the Software, unless such copies or derivative
// works are solely in the form of machine-executable object code generated by
// a source language processor.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
// SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
// FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.


#ifndef libpbe_charset_const_character_iterator_hh
#define libpbe_charset_const_character_iterator_hh

#include "charset_t.hh"
#include "charset_traits.hh"

#include <boost/iterator/iterator_adaptor.hpp>


namespace pbe {


// const_character_iterator
// ------------------------
//
// This iterator adaptor converts an iterator over the units of a string
// to an iterator over the characters.
// This is of course a trivial operation for fixed-length character sets,
// and non-trivial for variable-length ones.  The character-set-specific
// details are provided by various functions in the character set's traits
// class.
// The iterator is implemented using Boost's iterator_adaptor class.
//
// Typical use is as follows:
// std::container<char> ctr;    // some sort of container containing bytes,
//                              // in UTF-8.
// std::container::const_iterator i = ctr.begin();
//                              // i iterates over the bytes.
// const_character_iterator<cs::utf8, std::container::const_iterator> j = i;
//                              // j iterates over the decoded characters.
//
// The const_character_iterator is, as the name suggests, a const iterator:
// it's not possible to change the string using it since that could require
// that the string grows or shrinks.  It's also not a random-access iterator,
// even if the underlying iterator is random-access, since random access
// is O(n) rather than O(1).


template <charset_t cset, typename unit_iterator_t>
class const_character_iterator:
  public boost::iterator_adaptor<const_character_iterator<cset,unit_iterator_t>,
                                 unit_iterator_t,
                                 typename charset_traits<cset>::char_t const,
                                 boost::bidirectional_traversal_tag,
                                   // FIXME should be min(unit_iterator_t,bidirectional)
                                 typename charset_traits<cset>::char_t const>
{
  typedef charset_traits<cset> traits;
  typename traits::char_t dereference() const {
    unit_iterator_t i = const_character_iterator::iterator_adaptor_::base();
    return traits::decode(i);
  }
  void increment(void) {
    traits::skip_forward_char(const_character_iterator::iterator_adaptor_::base_reference());
  }
  void decrement(void) {
    traits::skip_backward_char(const_character_iterator::iterator_adaptor_::base_reference());
  }

  friend class boost::iterator_core_access;

public:
  const_character_iterator() {}
  const_character_iterator(const unit_iterator_t& iter): 
    const_character_iterator::iterator_adaptor_(iter)
  {}
};


};


#endif
