// include/flat_multimap.hh
// This file is part of libpbe; see http://svn.chezphil.org/libpbe/
// (C) 2010 Philip Endecott

// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#ifndef flat_multimap_hh
#define flat_multimap_hh

#include <vector>
#include <utility>
#include <memory>
#include <algorithm>
#include <functional>

// flat_multimap<K,D> is like a multimap<K,D> but implemented using 
// a vector<pair<K,D>> to save memory and perhaps increase speed. 
// The differences are:
// 
// - You must sort it, by calling the sort() method, between inserting 
//   elements and searching.
// - Iterator invalidation is like a vector, i.e. insertion invalidates 
//   if the capacity is exceeded, as does sorting.
// - There is no hint version of insert().
// - erase is O(N).
// - There are capacity() and reserve() methods.
//
// This implementation does not keep track of which parts of the vector 
// are already sorted.  If this is done, new elements can be merged 
// more efficiently.
//
// value_type should be pair<const key_type, data_type>, however I'm 
// not able to make this work; the const means that operator= doesn't 
// work, which the vector doesn't like.  So it's non-const instead.


namespace pbe {


template <typename COMP>
struct compare_first {
  COMP comp;
  compare_first(COMP comp_): comp(comp_) {}
  template <typename FIRST, typename SECOND>
  bool operator()(std::pair<FIRST,SECOND> a,
                  std::pair<FIRST,SECOND> b) const {
    return comp(a.first,b.first);
  }
};


template < typename KEY,
           typename DATA,
           typename COMPARE = std::less<KEY>,
           typename ALLOC = std::allocator< std::pair<KEY,DATA> >
         >
class flat_multimap {

public:
  typedef KEY     key_type;
  typedef DATA    data_type;
  typedef COMPARE key_compare;
  typedef std::pair</*const*/ key_type, data_type> value_type;  // FIXME
  typedef compare_first<key_compare> value_compare;

private:
  typedef std::vector<value_type,ALLOC> impl_t;
  impl_t impl;

  key_compare kcomp;
  value_compare vcomp;

public:
  typedef typename impl_t::pointer                pointer;
  typedef typename impl_t::reference              reference;
  typedef typename impl_t::const_reference        const_reference;
  typedef typename impl_t::size_type              size_type;
  typedef typename impl_t::difference_type        difference_type;
  typedef typename impl_t::iterator               iterator;
  typedef typename impl_t::const_iterator         const_iterator;
  typedef typename impl_t::reverse_iterator       reverse_iterator;
  typedef typename impl_t::const_reverse_iterator const_reverse_iterator;

  iterator               begin()        { return impl.begin(); }
  iterator               end()          { return impl.end(); }
  const_iterator         begin() const  { return impl.begin(); }
  const_iterator         end()   const  { return impl.end(); }
  reverse_iterator       rbegin()       { return impl.rbegin(); }
  reverse_iterator       rend()         { return impl.rend(); }
  const_reverse_iterator rbegin() const { return impl.rbegin(); }
  const_reverse_iterator rend()   const { return impl.rend(); }
  size_type              size()   const { return impl.size(); }
  size_type              max_size() const { return impl.max_size(); }
  size_type              capacity() const { return impl.capacity(); }
  bool                   empty()    const { return impl.empty(); }
  key_compare            key_comp() const { return kcomp; }
  //value_compare value_comp() const { return vcomp; }
  flat_multimap(): vcomp(kcomp) {}
  flat_multimap(const key_compare& comp): kcomp(comp), vcomp(kcomp) {}
  template <typename InputIterator>
  flat_multimap(InputIterator f, InputIterator l): impl(f,l), vcomp(kcomp) {}
  template <class InputIterator>
  flat_multimap(InputIterator f, InputIterator l, const key_compare& comp): impl(f,l), kcomp(comp), vcomp(kcomp) {}
  // flat_multimap(const flat_multimap&) = default
  // flat_multimap& operator=(const flat_multimap&) = default
  void reserve(size_t s) { impl.reserve(s); }
  void swap(flat_multimap& other) { swap(impl,other.impl); swap(kcomp,other.kcomp); }
  iterator insert(const value_type& x) { return impl.insert(impl.end(),x); }
  template <class InputIterator>
  void insert(InputIterator f, InputIterator l) { impl.insert(impl.end(),f,l); }
  void erase(iterator pos) { impl.erase(pos); }
  size_type erase(const key_type& k) { std::pair<iterator,iterator> r = std::equal_range(impl.begin(),impl.end(),value_type(k,data_type())); size_type d = r.second-r.first; impl.erase(r.first,r.second); return d; }
  void erase(iterator first, iterator last) { impl.erase(first,last); }
  void clear() { impl.clear(); }
  iterator find(const key_type& k) { iterator i = std::lower_bound(impl.begin(),impl.end(),value_type(k,data_type()),vcomp); return kcomp(k,i->first) ? impl.end() : i; }
  const_iterator find(const key_type& k) const { iterator i = std::lower_bound(impl.begin(),impl.end(),value_type(k,data_type()),vcomp); return kcomp(k,i->first) ? impl.end() : i; }
  size_type count(const key_type& k) { std::pair<iterator,iterator> r = equal_range(impl.begin(),impl.end(),value_type(k,data_type()),vcomp); return r.second-r.first; }
  iterator lower_bound(const key_type& k) { return std::lower_bound(impl.begin(),impl.end(),value_type(k,data_type()),vcomp); }
  const_iterator lower_bound(const key_type& k) const { return std::lower_bound(impl.begin(),impl.end(),value_type(k,data_type()),vcomp); }
  iterator upper_bound(const key_type& k) { return std::upper_bound(impl.begin(),impl.end(),value_type(k,data_type()),vcomp); }
  const_iterator upper_bound(const key_type& k) const { return std::upper_bound(impl.begin(),impl.end(),value_type(k,data_type()),vcomp); }
  std::pair<iterator,iterator> equal_range(const key_type& k) { return std::equal_range(impl.begin(),impl.end(),value_type(k,data_type()),vcomp); }
  std::pair<const_iterator,const_iterator> equal_range(const key_type& k) const { return std::equal_range(impl.begin(),impl.end(),value_type(k,data_type()),vcomp); }

  void sort() { std::sort(impl.begin(),impl.end(),vcomp); }

};


template < typename KEY, typename DATA, typename COMPARE, typename ALLOC >
bool operator==(const flat_multimap<KEY,DATA,COMPARE,ALLOC>& a,
                const flat_multimap<KEY,DATA,COMPARE,ALLOC>& b) {
  return a.impl == b.impl;
}

template < typename KEY, typename DATA, typename COMPARE, typename ALLOC >
bool operator<(const flat_multimap<KEY,DATA,COMPARE,ALLOC>& a,
                const flat_multimap<KEY,DATA,COMPARE,ALLOC>& b) {
  return a.impl < b.impl;
}


};


#endif

