// include/sbo_vector.hh
// This file is part of libpbe; see http://svn.chezphil.org/libpbe/
// (C) 2011 Philip Endecott

// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#ifndef libpbe_sbo_vector_hh
#define libpbe_sbo_vector_hh

#include "indexable_iter.hh"

#include <vector>


// sbo_vector<N,T> is like std::vector<T>, except that it has a small 
// buffer optimisation for up to N elements; that is, it stores up to N 
// elements on the stack, which is faster than storing them on the heap 
// as std::vector does.  The implementation falls back to std::vector 
// for the >N case.
// Due to lazyness, only a small subset of the std::vector functionality 
// is implemented.  I believe it should be possible to implement all of 
// std::vector if desired.
// Also due to lazyness, this version doesn't use a union to share the 
// same space for the vector and the sbo data.
// No particular attention has been paid to what might happen if an 
// exception is thrown, e.g. while copying from the SBO data into the 
// vector.
// indexable_iter is used to provide iterators.


namespace pbe {


template <typename T, size_t N>
class sbo_vector {

  bool using_vector;
  T data[N];
  size_t data_size;
  std::vector<T> vec;

  void clear_data()
  {
    for (T* p = data; p < data+data_size; ++p) {
      *p = T();
    }
  }

  void use_vector()
  {
    vec.insert(vec.end(),data,data+data_size);
    clear_data();
    using_vector = true;
  }

public:

  typedef T value_type;

  sbo_vector():
    using_vector(false), data_size(0)
  {}

  void clear()
  {
    if (using_vector) {
      vec.clear();
    } else {
      clear_data();
    }
    using_vector = false;
    data_size = 0;
  }

  void reserve(size_t sz)
  {
    if (sz>N) {
      vec.reserve(sz);
      if (!using_vector) {
        use_vector();
      }
    }
  }

  void push_back(const T& elem)
  {
    if (!using_vector && data_size==N) {
      use_vector();
    }
    if (using_vector) {
      vec.push_back(elem);
    } else {
      data[data_size] = elem;
      ++data_size;
    }
  }

  T& operator[](size_t i)
  {
    if (using_vector) return vec[i];
    else return data[i];
  }

  const T& operator[](size_t i) const
  {
    if (using_vector) return vec[i];
    else return data[i];
  }

  T& front() {
    return *begin();
  }

  const T& front() const {
    return *begin();
  }

  size_t size() const {
    return using_vector ? vec.size() : data_size;
  }

  bool empty() const {
    return using_vector ? vec.empty() : data_size==0;
  }

  typedef indexable_iter<sbo_vector> iterator;
  typedef const_indexable_iter<sbo_vector> const_iterator;

  iterator begin() { return iterator(this,0); }
  iterator end()   { return iterator(this,size()); }

  const_iterator begin() const { return const_iterator(this,0); }
  const_iterator end()   const { return const_iterator(this,size()); }

};


};


#endif



